//-----------------------------------------------------------------------------
// Microsoft OLE DB RowsetViewer
// Copyright (C) 1994 - 1998 By Microsoft Corporation.
//
// @doc
//
// @module DATASOURCE.CPP
//
//-----------------------------------------------------------------------------------

/////////////////////////////////////////////////////////////////
// Includes
//
/////////////////////////////////////////////////////////////////
#include "Common.h"
#include "DataSource.h"
#include "Main.h"


/////////////////////////////////////////////////////////////////
// CBase::CBase
//
/////////////////////////////////////////////////////////////////
CBase::CBase(CMDIChild* pCMDIChild)
{
	//BackPointer
	ASSERT(pCMDIChild);
	m_pCMDIChild =  pCMDIChild;
	
	//OLEDB Interfaces
	m_pIUnknown		= NULL;		
}


/////////////////////////////////////////////////////////////////
// CBase::~CBase
//
/////////////////////////////////////////////////////////////////
CBase::~CBase()
{
	ASSERT(m_pCMDIChild);
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	//Verify Object has no references on it...
	pCListBox->OutputRelease(&m_pIUnknown, "IUnknown", 0);
}


/////////////////////////////////////////////////////////////////
// BOOL CBase::ReleaseBase
//
/////////////////////////////////////////////////////////////////
HRESULT CBase::ReleaseBase(CHAR* pszObject)
{
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	//IUnknown interfaces
	//Shouldn't have any extra references left...
	pCListBox->OutputRelease(&m_pIUnknown, pszObject ? pszObject : "IUnknown", 0);
	return S_OK;
}


/////////////////////////////////////////////////////////////////
// ULONG CBase::AddRef
//
/////////////////////////////////////////////////////////////////
ULONG CBase::AddRef()
{
	//IUnknown::AddRef
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	return pCListBox->OutputAddRef(m_pIUnknown);
}

/////////////////////////////////////////////////////////////////
// ULONG CBase::Release
//
/////////////////////////////////////////////////////////////////
ULONG CBase::Release()
{
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	IUnknown* pIUnknown = m_pIUnknown;
	ULONG ulRefCount = 0;

	//IUnknown::Release
	ulRefCount = pCListBox->OutputRelease(&m_pIUnknown);
	if(ulRefCount != 0)
		m_pIUnknown = pIUnknown;

	return ulRefCount;
}


/////////////////////////////////////////////////////////////////
// HRESULT CBase::QueryInterface
//
/////////////////////////////////////////////////////////////////
HRESULT CBase::QueryInterface(REFIID riid)
{
	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	IUnknown* pIUnknown = NULL;
	
	//IUnknown::QueryInterface
	XTEST(hWnd, hr = pCListBox->OutputQI(m_pIUnknown, riid, &pIUnknown));
	
	//We need to put the obtained interface in the 
	//appropiate member or release it...
	if(FAILED(hr) || FAILED(SetInterface(riid, pIUnknown)))
		pCListBox->OutputRelease(&pIUnknown);

	return hr;
}


////////////////////////////////////////////////////////////////
// CBase::GetOptionsObj
//
/////////////////////////////////////////////////////////////////
COptionsDlg* CBase::GetOptionsObj()
{
	ASSERT(m_pCMDIChild);
	return m_pCMDIChild->GetOptionsObj();
}


/////////////////////////////////////////////////////////////////
// CDataSource::CDataSource
//
/////////////////////////////////////////////////////////////////
CDataSource::CDataSource(CMDIChild* pCMDIChild) : CBase(pCMDIChild)
{
	//OLEDB Interfaces
	m_pIDBCreateSession			= NULL;		//DataSource interface
	m_pIDBInitialize			= NULL;		//DataSource interface
	m_pIDBProperties			= NULL;		//DataSource interface
	m_pIPersist					= NULL;		//DataSource interface

	m_pIConnectionPointContainer= NULL;		//DataSource interface
	m_pIDBAsynchStatus			= NULL;		//DataSource interface
	m_pIDBDataSourceAdmin		= NULL;		//DataSource interface
	m_pIDBInfo					= NULL;		//DataSource interface
	m_pIPersistFile				= NULL;		//DataSource interface
	m_pISupportErrorInfo		= NULL;		//DataSource interface
			
	m_pISourcesRowset			= NULL;		//Enumerator interface
	m_pIParseDisplayName		= NULL;		//Enumerator interface
	
	//Extra interfaces
	m_pIConnectionPoint			= NULL;		//Connection interface

	//EnumInfo
	memset(&m_EnumInfo, 0, sizeof(ENUMINFO));
	
	//Properties
	m_pwszDataSource		= NULL;		//DBPROP_DATASOURCENAME
	m_pwszDBMS				= NULL;		//DBPROP_DBMSNAME
	m_pwszDBMSVer			= NULL;		//DBPROP_DBMSVER
	m_pwszProviderName		= NULL;
	m_pwszProviderDesc		= NULL;
	m_fConnected			= FALSE;

	//Listeners
	m_pCAsynchNotify		= new CAsynchNotify(m_pCMDIChild->m_pCListBox);
}

/////////////////////////////////////////////////////////////////
// CDataSource::~CDataSource
//
/////////////////////////////////////////////////////////////////
CDataSource::~CDataSource()
{
	ASSERT(m_pCMDIChild);
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	ReleaseDataSource();
	//Verify Listeners have no extra references on them...
	pCListBox->OutputRelease((IUnknown**)&m_pCAsynchNotify, "IDBAsynchNotify", 0);
}


/////////////////////////////////////////////////////////////////
// HRESULT CDataSource::SetInterface
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::SetInterface(REFIID riid, IUnknown* pIUnknown)
{
	HRESULT hr = S_OK;

	//no-op
	if(pIUnknown == NULL)
		return E_INVALIDARG;

	//If the requested interface is a DataSource interface, place it in the
	//Appropiate member variable, otherwise we have to release it, since we
	//have no place to store it...
	if(riid == IID_IUnknown) 
		m_pIUnknown = pIUnknown;
	else if(riid == IID_IDBCreateSession) 
		m_pIDBCreateSession = (IDBCreateSession*)pIUnknown;
	else if(riid == IID_IDBInitialize) 
		m_pIDBInitialize = (IDBInitialize*)pIUnknown;
	else if(riid == IID_IDBProperties) 
		m_pIDBProperties = (IDBProperties*)pIUnknown;
	else if(riid == IID_IPersist) 
		m_pIPersist = (IPersist*)pIUnknown;
	else if(riid == IID_IConnectionPointContainer) 
		m_pIConnectionPointContainer = (IConnectionPointContainer*)pIUnknown;
	else if(riid == IID_IDBAsynchStatus) 
		m_pIDBAsynchStatus = (IDBAsynchStatus*)pIUnknown;
	else if(riid == IID_IDBDataSourceAdmin) 
		m_pIDBDataSourceAdmin = (IDBDataSourceAdmin*)pIUnknown;
	else if(riid == IID_IDBInfo) 
		m_pIDBInfo = (IDBInfo*)pIUnknown;
	else if(riid == IID_IPersistFile) 
		m_pIPersistFile = (IPersistFile*)pIUnknown;
	else if(riid == IID_ISupportErrorInfo) 
		m_pISupportErrorInfo = (ISupportErrorInfo*)pIUnknown;
	else if(riid == IID_ISourcesRowset) 
		m_pISourcesRowset = (ISourcesRowset*)pIUnknown;
	else if(riid == IID_IParseDisplayName) 
		m_pIParseDisplayName = (IParseDisplayName*)pIUnknown;
	else
		hr = E_NOINTERFACE;

	return hr;
}


/////////////////////////////////////////////////////////////////
// BOOL CDataSource::ReleaseDataSource
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::ReleaseDataSource()
{
	//UnadviseListeners
	UnadviseListeners();

	//DataSource interfaces
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	pCListBox->OutputRelease((IUnknown**)&m_pIDBCreateSession,			"IDBCreateSession");
	pCListBox->OutputRelease((IUnknown**)&m_pIDBInitialize,				"IDBInitialize");
	pCListBox->OutputRelease((IUnknown**)&m_pIDBProperties,				"IDBProperties");
	pCListBox->OutputRelease((IUnknown**)&m_pIPersist,					"IPersist");

	pCListBox->OutputRelease((IUnknown**)&m_pIDBAsynchStatus,			"IDBAsynchStatus");
	pCListBox->OutputRelease((IUnknown**)&m_pIDBDataSourceAdmin,		"IDBDataSourceAdmin");
	pCListBox->OutputRelease((IUnknown**)&m_pIDBInfo,					"IDBInfo");
	pCListBox->OutputRelease((IUnknown**)&m_pIPersistFile,				"IPersistFile");
	pCListBox->OutputRelease((IUnknown**)&m_pISupportErrorInfo,			"ISupportErrorInfo");
	pCListBox->OutputRelease((IUnknown**)&m_pIConnectionPointContainer, "IConnectionPointContainer");

	//Enumerator interfaces
	pCListBox->OutputRelease((IUnknown**)&m_pISourcesRowset,			"ISourcesRowset");
	pCListBox->OutputRelease((IUnknown**)&m_pIParseDisplayName,			"IParseDisplayName");
	
	//Extra interfaces
	pCListBox->OutputRelease((IUnknown**)&m_pIConnectionPoint,			"IConnectionPoint");

	//Base
	ReleaseBase("DataSource");

	//Properties
	SAFE_FREE(m_pwszDataSource);
	SAFE_FREE(m_pwszDBMS);
	SAFE_FREE(m_pwszDBMSVer);
	SAFE_FREE(m_pwszProviderName);
	SAFE_FREE(m_pwszProviderDesc);

	//Data
	m_fConnected = FALSE;
	return S_OK;
}


/////////////////////////////////////////////////////////////////
// BOOL CDataSource::IsConnected
//
/////////////////////////////////////////////////////////////////
BOOL CDataSource::IsConnected()
{
	return m_fConnected;
}


/////////////////////////////////////////////////////////////////
// HRESULT CDataSource::CreateDataSource
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::CreateDataSource(CDataSource* pCDataSource)
{
	//Use exsiting Connection
	ASSERT(pCDataSource);
	HRESULT				hr = S_OK;
	HRESULT				hrOptional = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	//Release all Previous interfaces...
	IUnknown* pIUnknown = pCDataSource->m_pIUnknown;
	pCListBox->OutputAddRef(pIUnknown);
	ReleaseDataSource();
	m_pIUnknown = pIUnknown;

	//Determine the Type of this Object...
	hrOptional = pCListBox->OutputQI(m_pIUnknown,	IID_IParseDisplayName,	(IUnknown**)&m_pIParseDisplayName);
		
	//DataSource
	if(m_pIParseDisplayName == NULL)
	{
		//[MANDATORY]
		if(m_pCMDIChild->GetConnectObj()->m_dwConnectOpts & CONNECT_SETPROPERTIES || m_pCMDIChild->GetOptionsObj()->m_dwRowsetOpts & ROWSET_AUTOQI)
			XTEST(hWnd, hr = pCListBox->OutputQI(m_pIUnknown,	IID_IDBProperties,				(IUnknown**)&m_pIDBProperties));

		if(m_pCMDIChild->GetConnectObj()->m_dwConnectOpts & CONNECT_INITIALIZE	|| m_pCMDIChild->GetOptionsObj()->m_dwRowsetOpts & ROWSET_AUTOQI)
			XTESTC(hWnd, hr = pCListBox->OutputQI(m_pIUnknown,	IID_IDBInitialize,				(IUnknown**)&m_pIDBInitialize));
		
		if(m_pCMDIChild->GetConnectObj()->m_dwConnectOpts & CONNECT_CREATESESSION || m_pCMDIChild->GetOptionsObj()->m_dwRowsetOpts & ROWSET_AUTOQI)
			hrOptional = pCListBox->OutputQI(m_pIUnknown,		IID_IDBCreateSession,			(IUnknown**)&m_pIDBCreateSession);

		//AutoQI
		if(GetOptionsObj()->m_dwRowsetOpts & ROWSET_AUTOQI)
		{
			//[OPTIONAL]
			hrOptional = pCListBox->OutputQI(m_pIUnknown,		IID_IPersist,					(IUnknown**)&m_pIPersist);
			hrOptional = pCListBox->OutputQI(m_pIUnknown,		IID_IConnectionPointContainer,	(IUnknown**)&m_pIConnectionPointContainer);
			hrOptional = pCListBox->OutputQI(m_pIUnknown,		IID_IDBAsynchStatus,			(IUnknown**)&m_pIDBAsynchStatus);
			hrOptional = pCListBox->OutputQI(m_pIUnknown,		IID_IDBDataSourceAdmin,			(IUnknown**)&m_pIDBDataSourceAdmin);
			hrOptional = pCListBox->OutputQI(m_pIUnknown,		IID_IDBInfo,					(IUnknown**)&m_pIDBInfo);
			hrOptional = pCListBox->OutputQI(m_pIUnknown,		IID_IPersistFile,				(IUnknown**)&m_pIPersistFile);
			hrOptional = pCListBox->OutputQI(m_pIUnknown,		IID_ISupportErrorInfo,			(IUnknown**)&m_pISupportErrorInfo);
		}

		//DBPROP_DATASOURCENAME
		GetProperty(m_pIUnknown, DBPROP_DATASOURCENAME,			DBPROPSET_DATASOURCEINFO, &m_pwszDataSource);
		//DBPROP_DBMSNAME
		GetProperty(m_pIUnknown, DBPROP_DBMSNAME,				DBPROPSET_DATASOURCEINFO, &m_pwszDBMS);
		//DBPROP_DBMSVER
		GetProperty(m_pIUnknown, DBPROP_DBMSVER,				DBPROPSET_DATASOURCEINFO, &m_pwszDBMSVer);
		//DBPROP_PROVIDERFILENAME
		GetProperty(m_pIUnknown, DBPROP_PROVIDERFILENAME,		DBPROPSET_DATASOURCEINFO, &m_pwszProviderName);
		//DBPROP_PROVIDERFRIENDLYNAME
		GetProperty(m_pIUnknown, DBPROP_PROVIDERFRIENDLYNAME,	DBPROPSET_DATASOURCEINFO, &m_pwszProviderDesc);
	}
	else //Enumerator
	{
		//AutoQI
		if(GetOptionsObj()->m_dwRowsetOpts & ROWSET_AUTOQI)
		{
			//[MANDATORY]
			XTESTC(hWnd, hr = pCListBox->OutputQI(m_pIUnknown,	IID_ISourcesRowset,		(IUnknown**)&m_pISourcesRowset));
			
			//[OPTIONAL]
			hrOptional = pCListBox->OutputQI(m_pIUnknown,	IID_IDBInitialize,			(IUnknown**)&m_pIDBInitialize);
			hrOptional = pCListBox->OutputQI(m_pIUnknown,	IID_IDBProperties,			(IUnknown**)&m_pIDBProperties);
			hrOptional = pCListBox->OutputQI(m_pIUnknown,	IID_ISupportErrorInfo,		(IUnknown**)&m_pISupportErrorInfo);
		}
	}

	//Copy EnumInfo
	memcpy(&m_EnumInfo, &pCDataSource->m_EnumInfo, sizeof(ENUMINFO));

CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CDataSource::AdviseListeners
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::AdviseListeners()
{
	HRESULT hr = E_FAIL;
	
	//IDBAsynchNotify Listener
	//This is optional functionality so don't post error
	if(m_pIUnknown && GetOptionsObj()->m_dwNotifyOpts & NOTIFY_IDBASYNCHNOTIFY)
		hr = m_pCAsynchNotify->Advise(m_pIUnknown);

	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CDataSource::UnadviseListeners
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::UnadviseListeners()
{
	HRESULT hr = E_FAIL;
	
	//IDBAsynchNotify Listener
	//This is optional functionality so don't post error
	if(m_pIUnknown)
		hr = m_pCAsynchNotify->Unadvise(m_pIUnknown);

	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CDataSource::CreateDataSource
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::CreateDataSource(IUnknown* pIUnknown, ULONG cPropSets, DBPROPSET* rgPropSets)
{
	ASSERT(pIUnknown);
	HRESULT			hr = E_FAIL;

	//Connect to the provider
	TESTC(hr = Connect(pIUnknown));

	//SetProperties
	TESTC(hr = SetProperties(cPropSets, rgPropSets));

	//Initialize
	TESTC(hr = Initialize());

CLEANUP:
	return hr;
}



/////////////////////////////////////////////////////////////////
// HRESULT CDataSource::Connect
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::Connect(IUnknown* pIUnknown)
{
	//No-op
	if(pIUnknown == NULL)
		return E_FAIL;
	
	HWND	hWnd = m_pCMDIChild->m_hWnd;		
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	ASSERT(m_pIDBInitialize == NULL);
	ASSERT(m_pIDBProperties	== NULL);

	pCListBox->OutputAddRef(pIUnknown);
	m_pIUnknown = pIUnknown;

	//The Object (Enum or DataSource) may require initialization
	pCListBox->OutputQI(m_pIUnknown, IID_IDBInitialize, (IUnknown**)&m_pIDBInitialize);
	pCListBox->OutputQI(m_pIUnknown, IID_IDBProperties, (IUnknown**)&m_pIDBProperties);
	return S_OK;
}


/////////////////////////////////////////////////////////////////
// HRESULT CDataSource::Initialize
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::Initialize()
{
	HRESULT	hr = S_OK;
	HWND	hWnd = m_pCMDIChild->m_hWnd;		
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	if(m_pIDBInitialize)
	{
		//We actually want to advise our listeners, before Initialization
		//So we can get any notifications that occurr....
		AdviseListeners();

		//Initailize
		pCListBox->OutputPreMethod("IDBInitialize::Initialize()");
		XTEST_(hWnd, hr = m_pIDBInitialize->Initialize(),S_OK);
		pCListBox->OutputPostMethod(hr, "IDBInitialize::Initialize()");
		if(hr == DB_S_ERRORSOCCURRED || hr == DB_E_ERRORSOCCURRED)
		{
			DisplayPropErrors(hWnd, DATASOURCE, m_pIDBInitialize);

			//We don't want to continue if there were property errors
			//during intialization, treat as if a failure...
			hr = DB_E_ERRORSOCCURRED;
			goto CLEANUP;
		}
		TESTC(hr);
	}

	//Delegate to other Create which assumes already Initialized
	TESTC(hr = CreateDataSource(this));

	//We are now Connected and Initialized
	m_fConnected = TRUE;
	
CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CDataSource::SetProperties
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::SetProperties(ULONG cPropSets, DBPROPSET* rgPropSets)
{
	HRESULT	hr = S_OK;
	HWND	hWnd = m_pCMDIChild->m_hWnd;		
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	if(m_pIDBProperties && cPropSets)
	{
		//SetProperties
		pCListBox->OutputPreMethod("IDBProperties::SetProperties(%d, 0x%08x)", cPropSets, rgPropSets);
		XTEST_(hWnd, hr = m_pIDBProperties->SetProperties(cPropSets, rgPropSets),S_OK);
		pCListBox->OutputPostMethod(hr, "IDBProperties::SetProperties(%d, 0x%08x)", cPropSets, rgPropSets);
		if(hr == DB_S_ERRORSOCCURRED || hr == DB_E_ERRORSOCCURRED)
		{
			DisplayPropErrors(hWnd, cPropSets, rgPropSets);

			//We don't want to continue if there were property errors
			//during intialization, treat as if a failure...
			hr = DB_E_ERRORSOCCURRED;
			goto CLEANUP;
		}
		TESTC(hr);
	}

CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CDataSource::SetEnumInfo
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::SetEnumInfo(ENUMINFO* pEnumInfo)
{
	HRESULT hr = S_OK;
	WCHAR* pwszProgID = NULL;
	CLSID clsid;
	
	if(pEnumInfo)
	{
		//This is additional info is from the Root Enumerator
		//Contains info such as Name, ParseName, Description
		//Of the provider, mainly only used for UI purposes, but nice to have incase...
		memcpy(&m_EnumInfo, pEnumInfo, sizeof(ENUMINFO));
	}
	else
	{
		memset(&m_EnumInfo, 0, sizeof(ENUMINFO));
		TESTC(hr = GetClassID(clsid, &pwszProgID));
		if(pwszProgID)
		{
			wcscpy(m_EnumInfo.wszName, pwszProgID);
			wcscpy(m_EnumInfo.wszParseName, pwszProgID);
		}
		
	}
		
CLEANUP:
	SAFE_FREE(pwszProgID);
	return S_OK;
}

////////////////////////////////////////////////////////////////
// CDataSource::GetEnumInfo
//
/////////////////////////////////////////////////////////////////
ENUMINFO* CDataSource::GetEnumInfo()
{
	return &m_EnumInfo;
}


////////////////////////////////////////////////////////////////
// CDataSource::GetClassID
//
/////////////////////////////////////////////////////////////////
HRESULT CDataSource::GetClassID(REFCLSID clsid, WCHAR** ppwszProgID)
{
	ASSERT(m_pIPersist);
	HRESULT hr = S_OK;
	HWND	hWnd = m_pCMDIChild->m_hWnd;		
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	WCHAR* pwszProgID = NULL;

	//IPersist::GetClassID
	TESTC(pCListBox->OutputPreMethod("IPerist::GetClassID(%S)", pwszProgID));
	XTEST(hWnd, hr = m_pIPersist->GetClassID((CLSID*)&clsid));

	//Obtain ProgID
	pwszProgID = GetProgID(clsid);
	TESTC(pCListBox->OutputPostMethod(hr, "IPerist::GetClassID(%S)", pwszProgID));
	
CLEANUP:
	if(ppwszProgID)
		*ppwszProgID = pwszProgID;
	else
		SAFE_FREE(pwszProgID);
	return hr;
}



/////////////////////////////////////////////////////////////////
// CSession::CSession
//
/////////////////////////////////////////////////////////////////
CSession::CSession(CMDIChild* pCMDIChild, CDataSource* pCDataSource) : CBase(pCMDIChild)
{
	//BackPointer
	ASSERT(pCDataSource);
	m_pCDataSource =  pCDataSource;
	
	//OLEDB Interfaces
	m_pIGetDataSource		= NULL;		//Session interface
	m_pIOpenRowset			= NULL;		//Session interface
	m_pISessionProperties	= NULL;		//Session interface
	m_pIDBCreateCommand		= NULL;		//Session interface
	m_pIDBSchemaRowset		= NULL;		//Session interface
	m_pIIndexDefinition		= NULL;		//Session interface
	m_pISupportErrorInfo	= NULL;		//Session interface
	m_pITableDefinition		= NULL;		//Session interface
	m_pITransactionJoin		= NULL;		//Session interface
	m_pITransactionLocal	= NULL;		//Session interface
	m_pITransactionObject	= NULL;		//Session interface

	//Objects
	m_pCTransaction	= new CTransaction(pCMDIChild);
}

/////////////////////////////////////////////////////////////////
// CSession::~CSession
//
/////////////////////////////////////////////////////////////////
CSession::~CSession()
{
	//Objects
	SAFE_DELETE(m_pCTransaction);

	ReleaseSession();
}


/////////////////////////////////////////////////////////////////
// HRESULT CSession::SetInterface
//
/////////////////////////////////////////////////////////////////
HRESULT CSession::SetInterface(REFIID riid, IUnknown* pIUnknown)
{
	HRESULT hr = S_OK;

	//no-op
	if(pIUnknown == NULL)
		return E_INVALIDARG;

	//If the requested interface is a Session interface, place it in the
	//Appropiate member variable, otherwise we have to release it, since we
	//have no place to store it...
	if(riid == IID_IGetDataSource) 
		m_pIGetDataSource = (IGetDataSource*)pIUnknown;
	else if(riid == IID_IOpenRowset) 
		m_pIOpenRowset = (IOpenRowset*)pIUnknown;
	else if(riid == IID_ISessionProperties) 
		m_pISessionProperties = (ISessionProperties*)pIUnknown;
	else if(riid == IID_IDBCreateCommand) 
		m_pIDBCreateCommand = (IDBCreateCommand*)pIUnknown;
	else if(riid == IID_IDBSchemaRowset) 
		m_pIDBSchemaRowset = (IDBSchemaRowset*)pIUnknown;
	else if(riid == IID_IIndexDefinition) 
		m_pIIndexDefinition = (IIndexDefinition*)pIUnknown;
	else if(riid == IID_ISupportErrorInfo) 
		m_pISupportErrorInfo = (ISupportErrorInfo*)pIUnknown;
	else if(riid == IID_ITableDefinition) 
		m_pITableDefinition = (ITableDefinition*)pIUnknown;
	else if(riid == IID_ITransactionJoin) 
		m_pITransactionJoin = (ITransactionJoin*)pIUnknown;
	else if(riid == IID_ITransactionLocal) 
		m_pITransactionLocal = (ITransactionLocal*)pIUnknown;
	else if(riid == IID_ITransactionObject) 
		m_pITransactionObject = (ITransactionObject*)pIUnknown;
	else
		hr = E_NOINTERFACE;

	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CSession::ReleaseSession
//
/////////////////////////////////////////////////////////////////
HRESULT CSession::ReleaseSession()
{
	//Session
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	pCListBox->OutputRelease((IUnknown**)&m_pIGetDataSource,		"IGetDataSource");
	pCListBox->OutputRelease((IUnknown**)&m_pIOpenRowset,			"IOpenRowset");
	pCListBox->OutputRelease((IUnknown**)&m_pISessionProperties,	"ISessionProperties");
	pCListBox->OutputRelease((IUnknown**)&m_pIDBCreateCommand,		"IDBCreateCommand");
	pCListBox->OutputRelease((IUnknown**)&m_pIDBSchemaRowset,		"IDBSchemaRowset");
	pCListBox->OutputRelease((IUnknown**)&m_pIIndexDefinition,		"IIndexDefinition");
	pCListBox->OutputRelease((IUnknown**)&m_pISupportErrorInfo,		"ISupportErrorInfo");
	pCListBox->OutputRelease((IUnknown**)&m_pITableDefinition,		"ITableDefinition");
	pCListBox->OutputRelease((IUnknown**)&m_pITransactionJoin,		"ITransactionJoin");
	pCListBox->OutputRelease((IUnknown**)&m_pITransactionLocal,		"ITransactionLocal");
	pCListBox->OutputRelease((IUnknown**)&m_pITransactionObject,	"ITransactionObject");
	ReleaseBase("Session");
	return S_OK;
}


/////////////////////////////////////////////////////////////////
// HRESULT CSession::CreateSession
//
/////////////////////////////////////////////////////////////////
HRESULT CSession::CreateSession(CSession* pCSession)
{
	//Use existing Session
	ASSERT(pCSession);
	HRESULT				hr = S_OK;
	HRESULT				hrOptional = S_OK;

	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	HWND hWnd = pCListBox->m_hWnd;		

	//Release all Previous interfaces...
	IUnknown* pIUnknown = pCSession->m_pIUnknown;
	pCListBox->OutputAddRef(pIUnknown);
	ReleaseSession();
	m_pIUnknown = pIUnknown;
	
	//[OPTIONAL]
	if(m_pCMDIChild->GetConnectObj()->m_dwConnectOpts & CONNECT_CREATECOMMAND || m_pCMDIChild->GetOptionsObj()->m_dwRowsetOpts & ROWSET_AUTOQI)
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IDBCreateCommand,		(IUnknown**)&m_pIDBCreateCommand);

	//AutoQI
	if(GetOptionsObj()->m_dwRowsetOpts & ROWSET_AUTOQI)
	{  	
		//[MANDATORY] - We might be able to continue without these mandatory interface...
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IOpenRowset,			(IUnknown**)&m_pIOpenRowset);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IGetDataSource,		(IUnknown**)&m_pIGetDataSource);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ISessionProperties,	(IUnknown**)&m_pISessionProperties);

		//[OPTIONAL]
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IDBSchemaRowset,		(IUnknown**)&m_pIDBSchemaRowset);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IIndexDefinition,		(IUnknown**)&m_pIIndexDefinition);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ISupportErrorInfo,	(IUnknown**)&m_pISupportErrorInfo);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ITableDefinition,		(IUnknown**)&m_pITableDefinition);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ITransactionJoin,		(IUnknown**)&m_pITransactionJoin);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ITransactionLocal,	(IUnknown**)&m_pITransactionLocal);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ITransactionObject,	(IUnknown**)&m_pITransactionObject);
	}

//CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CSession::CreateSession
//
/////////////////////////////////////////////////////////////////
HRESULT CSession::CreateSession(REFIID riid)
{
	HRESULT				hr = S_OK;
	HRESULT				hrOptional = S_OK;

	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	HWND hWnd = pCListBox->m_hWnd;		
	IUnknown* pIUnknown = NULL;

	//This interface is required to continue...
	if(m_pCDataSource->m_pIDBCreateSession)
	{
		//CreateSession
		TESTC(pCListBox->OutputPreMethod("IDBCreateSession::CreateSession(NULL, %s, &0x%08x)", GetInterfaceName(riid), pIUnknown));
		XTEST(hWnd, hr = m_pCDataSource->m_pIDBCreateSession->CreateSession(NULL, riid, (IUnknown**)&pIUnknown));
		TESTC(pCListBox->OutputPostMethod(hr, "IDBCreateSession::CreateSession(NULL, %s, &0x%08x)", GetInterfaceName(riid), pIUnknown));

		//Now that we now CreateSession has succeeded, use that for our 
		//object, otherwise we ruin our existing session if there was one...
		m_pIUnknown = pIUnknown;

		//Delegate
		TESTC(hr = CreateSession(this));
	}

CLEANUP:
	return hr;
}




/////////////////////////////////////////////////////////////////
// CCommand::CCommand
//
/////////////////////////////////////////////////////////////////
CCommand::CCommand(CMDIChild* pCMDIChild, CSession* pCSession) : CBase(pCMDIChild)
{
	//BackPointer
	ASSERT(pCSession);
	m_pCSession =  pCSession;
	
	//OLEDB Interfaces
	m_pIAccessor				= NULL;		//Command interface
	m_pIColumnsInfo				= NULL;		//Command interface
	m_pICommand					= NULL;		//Command interface
	m_pICommandProperties		= NULL;		//Command interface
	m_pICommandText				= NULL;		//Command interface
	m_pIConvertType				= NULL;		//Command interface
	m_pIColumnsRowset			= NULL;		//Command interface
	m_pICommandPrepare			= NULL;		//Command interface
	m_pICommandWithParameters	= NULL;		//Command interface
	m_pISupportErrorInfo		= NULL;		//Command interface

	//Parameters
	m_cParamBindings = 0;
	m_rgParamBindings = NULL;
	memset(&m_ParamInfo, 0, sizeof(DBPARAMS));
}

/////////////////////////////////////////////////////////////////
// CCommand::~CCommand
//
/////////////////////////////////////////////////////////////////
CCommand::~CCommand()
{
	ReleaseCommand();
}


/////////////////////////////////////////////////////////////////
// HRESULT CCommand::SetInterface
//
/////////////////////////////////////////////////////////////////
HRESULT CCommand::SetInterface(REFIID riid, IUnknown* pIUnknown)
{
	HRESULT hr = S_OK;

	//no-op
	if(pIUnknown == NULL)
		return E_INVALIDARG;

	//If the requested interface is a Command interface, place it in the
	//Appropiate member variable, otherwise we have to release it, since we
	//have no place to store it...
	if(riid == IID_IAccessor) 
		m_pIAccessor = (IAccessor*)pIUnknown;
	else if(riid == IID_IColumnsInfo) 
		m_pIColumnsInfo = (IColumnsInfo*)pIUnknown;
	else if(riid == IID_ICommand) 
		m_pICommand = (ICommand*)pIUnknown;
	else if(riid == IID_ICommandProperties) 
		m_pICommandProperties = (ICommandProperties*)pIUnknown;
	else if(riid == IID_ICommandText) 
		m_pICommandText = (ICommandText*)pIUnknown;
	else if(riid == IID_IConvertType) 
		m_pIConvertType = (IConvertType*)pIUnknown;
	else if(riid == IID_IColumnsRowset) 
		m_pIColumnsRowset = (IColumnsRowset*)pIUnknown;
	else if(riid == IID_ICommandPrepare) 
		m_pICommandPrepare = (ICommandPrepare*)pIUnknown;
	else if(riid == IID_ICommandWithParameters) 
		m_pICommandWithParameters = (ICommandWithParameters*)pIUnknown;
	else if(riid == IID_ISupportErrorInfo) 
		m_pISupportErrorInfo = (ISupportErrorInfo*)pIUnknown;
	else
		hr = E_NOINTERFACE;

	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CCommand::ReleaseCommand
//
/////////////////////////////////////////////////////////////////
HRESULT CCommand::ReleaseCommand()
{
	ReleaseAccessor(&m_ParamInfo.hAccessor);
	FreeBindings(&m_cParamBindings, &m_rgParamBindings);
	SAFE_FREE(m_ParamInfo.pData);

	//Command
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	pCListBox->OutputRelease((IUnknown**)&m_pIAccessor,				"IAccessor");
	pCListBox->OutputRelease((IUnknown**)&m_pIColumnsInfo,			"IColumnsInfo");
	pCListBox->OutputRelease((IUnknown**)&m_pICommand,				"ICommand");
	pCListBox->OutputRelease((IUnknown**)&m_pICommandProperties,	"ICommandProperties");
	pCListBox->OutputRelease((IUnknown**)&m_pICommandText,			"ICommandText");
	pCListBox->OutputRelease((IUnknown**)&m_pIConvertType,			"IConvertType");
	pCListBox->OutputRelease((IUnknown**)&m_pIColumnsRowset,		"IColumnsRowset");
	pCListBox->OutputRelease((IUnknown**)&m_pICommandPrepare,		"ICommandPrepare");
	pCListBox->OutputRelease((IUnknown**)&m_pICommandWithParameters,"ICommandWithParameters");
	pCListBox->OutputRelease((IUnknown**)&m_pISupportErrorInfo,		"ISupportErrorInfo");
	ReleaseBase("Command");
	return S_OK;
}


/////////////////////////////////////////////////////////////////
// HRESULT CCommand::CreateCommand
//
/////////////////////////////////////////////////////////////////
HRESULT CCommand::CreateCommand(REFIID riid)
{
	HRESULT				hr = S_OK;
	HRESULT				hrOptional = S_OK;
	
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	HWND hWnd = pCListBox->m_hWnd;
	IUnknown* pIUnknown = NULL;

	if(m_pCSession->m_pIDBCreateCommand)
	{
		//CreateCommand
		TESTC(pCListBox->OutputPreMethod("IDBCreateCommand::CreateCommand(NULL, %s, &0x%08x", GetInterfaceName(riid), pIUnknown));
		XTEST(hWnd, hr = m_pCSession->m_pIDBCreateCommand->CreateCommand(NULL, riid, (IUnknown**)&pIUnknown));
		TESTC(pCListBox->OutputPostMethod(hr, "IDBCreateCommand::CreateCommand(NULL, %s, &0x%08x", GetInterfaceName(riid), pIUnknown));

		//Now that we now CreateSession has succeeded, use that for our 
		//object, otherwise we ruin our existing session if there was one...
		ReleaseCommand();
		m_pIUnknown = pIUnknown;

		//AutoQI
		if(GetOptionsObj()->m_dwRowsetOpts & ROWSET_AUTOQI)
		{
			//[MANDATORY]
			XTESTC(hWnd, hr = pCListBox->OutputQI(m_pIUnknown, IID_ICommand,			(IUnknown**)&m_pICommand));
			XTESTC(hWnd, hr = pCListBox->OutputQI(m_pIUnknown, IID_IAccessor,			(IUnknown**)&m_pIAccessor));
			XTESTC(hWnd, hr = pCListBox->OutputQI(m_pIUnknown, IID_IColumnsInfo,		(IUnknown**)&m_pIColumnsInfo));
			XTESTC(hWnd, hr = pCListBox->OutputQI(m_pIUnknown, IID_ICommandText,		(IUnknown**)&m_pICommandText));

			//[OPTIONAL]
			hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ICommandProperties,		(IUnknown**)&m_pICommandProperties);
			hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IConvertType,				(IUnknown**)&m_pIConvertType);
			hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IColumnsRowset,			(IUnknown**)&m_pIColumnsRowset);
			hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ICommandPrepare,			(IUnknown**)&m_pICommandPrepare);
			hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ICommandWithParameters,	(IUnknown**)&m_pICommandWithParameters);
			hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ISupportErrorInfo,		(IUnknown**)&m_pISupportErrorInfo);
		}
	}

CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CCommand::CreateParamAccessor
//
/////////////////////////////////////////////////////////////////
HRESULT CCommand::CreateParamAccessor(ULONG cParams, DBPARAMINFO* rgParamInfo)
{
	HRESULT hr = S_OK;
	ULONG cbRowSize = 0;
	
	//Release Previous Accessor
	ReleaseAccessor(&m_ParamInfo.hAccessor);
	FreeBindings(&m_cParamBindings, &m_rgParamBindings);
	SAFE_FREE(m_ParamInfo.pData);

	//Delegate
	TESTC(hr = SetupBindings(cParams, rgParamInfo, &m_cParamBindings, &m_rgParamBindings, &cbRowSize));
	TESTC(hr = CreateAccessor(DBACCESSOR_PARAMETERDATA, m_cParamBindings, m_rgParamBindings, cbRowSize, &m_ParamInfo.hAccessor));
	
	//Allocate pData
	SAFE_ALLOC(m_ParamInfo.pData, BYTE, cbRowSize);

CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CCommand::SetupBindings
//
/////////////////////////////////////////////////////////////////
HRESULT CCommand::SetupBindings(ULONG cParams, DBPARAMINFO* rgParamInfo, ULONG* pcBindings, DBBINDING** prgBindings, ULONG* pcRowSize, BOOL* pbOutofLine)
{
	ASSERT(pcBindings);
	ASSERT(prgBindings);

	HRESULT hr = S_OK;
	CMDIChild* pCMDIChild = m_pCMDIChild;
	HWND hWnd = pCMDIChild->m_hWnd;
	CListBox* pCListBox = pCMDIChild->m_pCListBox;

	ULONG dwOffset = 0;
	ULONG i,cBindings = 0;
	DBBINDING* rgBindings = NULL;
	ULONG cStorageObjects = 0;
	
	//Only capable of the Following Converions (for Display)
	DBTYPE wBindingType = (DBTYPE)GetOptionsObj()->m_dwBindingType;

	//Alloc the space to hold the Bindings and Accessors
	SAFE_ALLOC(rgBindings, DBBINDING, cParams);

	cBindings = 0;
	for(i=0; i<cParams; i++) 
	{
		ASSERT(rgParamInfo);

		//SetUp the Bindings
		rgBindings[cBindings].iOrdinal	= rgParamInfo[i].iOrdinal;
		rgBindings[cBindings].obStatus	= dwOffset;
		rgBindings[cBindings].obLength	= dwOffset + sizeof(DBSTATUS);
		rgBindings[cBindings].obValue	= dwOffset + sizeof(DBSTATUS) + sizeof(ULONG);
		
		rgBindings[cBindings].pTypeInfo = NULL;
		rgBindings[cBindings].pBindExt  = NULL;

		rgBindings[cBindings].dwPart	= DBPART_VALUE|DBPART_LENGTH|DBPART_STATUS;			
		rgBindings[cBindings].dwMemOwner= DBMEMOWNER_CLIENTOWNED;
		
		rgBindings[cBindings].eParamIO	= DBPARAMIO_NOTPARAM;
		if(rgParamInfo[i].dwFlags & DBPARAMFLAGS_ISINPUT)
			rgBindings[cBindings].eParamIO |= DBPARAMIO_INPUT;
		if(rgParamInfo[i].dwFlags & DBPARAMFLAGS_ISOUTPUT)
			rgBindings[cBindings].eParamIO |= DBPARAMIO_OUTPUT;
		
		rgBindings[cBindings].dwFlags	= 0;
		rgBindings[cBindings].bPrecision= rgParamInfo[i].bPrecision;
		rgBindings[cBindings].bScale	= rgParamInfo[i].bScale;

		rgBindings[cBindings].pObject	= NULL;
		rgBindings[cBindings].wType		= wBindingType;

		//MAX_LENGTH
		rgBindings[cBindings].cbMaxLen	= rgParamInfo[i].ulParamSize;

		//Account for TYPE -> VARIANT Converion
		if(wBindingType == DBTYPE_VARIANT)
		{
			rgBindings[cBindings].cbMaxLen = sizeof(VARIANT);
		}
		
		//Account for the NULL terminator
		if(rgBindings[cBindings].wType == DBTYPE_STR)
			rgBindings[cBindings].cbMaxLen	+= sizeof(CHAR);
		if(rgBindings[cBindings].wType == DBTYPE_WSTR)
			rgBindings[cBindings].cbMaxLen	+= sizeof(WCHAR);

		//MAX_LENGTH
		rgBindings[cBindings].cbMaxLen	= min(rgBindings[cBindings].cbMaxLen, MAX_COL_SIZE);
		dwOffset = rgBindings[cBindings].cbMaxLen + rgBindings[cBindings].obValue;
		dwOffset = ROUNDUP( dwOffset );
		cBindings++;
	}

	//Size for pData
	if(pcRowSize)
		*pcRowSize = dwOffset;

CLEANUP:
	//Accessors
	*pcBindings = cBindings;
	*prgBindings = rgBindings;
	return hr;
}



/////////////////////////////////////////////////////////////////
// HRESULT CCommand::CreateAccessor
//
/////////////////////////////////////////////////////////////////
HRESULT CCommand::CreateAccessor(DBACCESSORFLAGS dwAccessorFlags, ULONG cBindings, DBBINDING* rgBindings, ULONG cRowSize, HACCESSOR* phAccessor)
{
	ASSERT(phAccessor);
	ASSERT(m_pIAccessor);

	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	DBBINDSTATUS* rgStatus = NULL;

	//Alloc the space to hold the status
	SAFE_ALLOC(rgStatus, DBBINDSTATUS, cBindings);

	//Create the accessor
	*phAccessor = NULL;
	TESTC(pCListBox->OutputPreMethod("IAccessor::CreateAccessor(%d, %d, 0x%08x, %d, &0x%08x, 0x%08x)", dwAccessorFlags, cBindings, rgBindings, cRowSize, *phAccessor, rgStatus));
	XTEST(hWnd, hr = m_pIAccessor->CreateAccessor(dwAccessorFlags, cBindings, rgBindings, cRowSize, phAccessor, rgStatus));
	if(hr == DB_S_ERRORSOCCURRED || hr == DB_E_ERRORSOCCURRED)
		DisplayAccessorErrors(NULL, cBindings, rgBindings, rgStatus);
	TESTC(pCListBox->OutputPostMethod(hr, "IAccessor::CreateAccessor(%d, %d, 0x%08x, 0, &0x%08x, 0x%08x)", dwAccessorFlags, cBindings, rgBindings, cRowSize, *phAccessor, rgStatus));

CLEANUP:
	SAFE_FREE(rgStatus);
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CCommand::ReleaseAccessor
//
/////////////////////////////////////////////////////////////////
HRESULT CCommand::ReleaseAccessor(HACCESSOR* phAccessor)
{
	ASSERT(phAccessor);
	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	ULONG ulRefCount = 0;

	if(m_pIAccessor && *phAccessor)
	{
		TESTC(hr = pCListBox->OutputPreMethod("IAccessor::ReleaseAccessor(0x%08x, &%d)", *phAccessor, ulRefCount));
		XTEST(hWnd, hr = m_pIAccessor->ReleaseAccessor(*phAccessor, &ulRefCount));
		TESTC(hr = pCListBox->OutputPostMethod(hr, "IAccessor::ReleaseAccessor(0x%08x, &%d)", *phAccessor, ulRefCount));
	}

CLEANUP:
	//Only NULL the handle if the RefCount is 0
	if(ulRefCount == 0)
		*phAccessor = NULL;
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CCommand::SetCommandText
//
/////////////////////////////////////////////////////////////////
HRESULT CCommand::SetCommandText(WCHAR* pwszText)
{
	HRESULT hr = S_OK;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	COptionsDlg* pCOptionsDlg = GetOptionsObj();
	HWND hWnd = m_pCMDIChild->m_hWnd;

	//Set CommandText
	if(m_pICommandText)
	{
		TESTC(pCListBox->OutputPreMethod("ICommandText::SetCommandText(%S, \"%S\")", pCOptionsDlg->m_wszDialect, pwszText));
		XTEST(hWnd, hr = m_pICommandText->SetCommandText(pCOptionsDlg->m_guidDialect, pwszText));
		TESTC(pCListBox->OutputPostMethod(hr, "ICommandText::SetCommandText(%S, \"%S\")", pCOptionsDlg->m_wszDialect, pwszText));
	}

CLEANUP:
	return hr;
}




/////////////////////////////////////////////////////////////////
// CTransaction::CTransaction
//
/////////////////////////////////////////////////////////////////
CTransaction::CTransaction(CMDIChild* pCMDIChild) : CBase(pCMDIChild)
{
	//OLEDB Interfaces
	m_pITransaction				= NULL;		//Transaction interface
	m_pIConnectionPointContainer= NULL;		//Transaction interface
	m_pISupportErrorInfo		= NULL;		//Transaction interface
	
	//Extra interfaces
	m_pITransactionOptions		= NULL;		//TransactionOptions interface
}

/////////////////////////////////////////////////////////////////
// CTransaction::~CTransaction
//
/////////////////////////////////////////////////////////////////
CTransaction::~CTransaction()
{
	ReleaseObject();
}


/////////////////////////////////////////////////////////////////
// HRESULT CTransaction::SetInterface
//
/////////////////////////////////////////////////////////////////
HRESULT CTransaction::SetInterface(REFIID riid, IUnknown* pIUnknown)
{
	HRESULT hr = S_OK;

	//no-op
	if(pIUnknown == NULL)
		return E_INVALIDARG;

	//If the requested interface is a DataSource interface, place it in the
	//Appropiate member variable, otherwise we have to release it, since we
	//have no place to store it...
	if(riid == IID_IUnknown) 
		m_pIUnknown = pIUnknown;
	else if(riid == IID_ITransaction) 
		m_pITransaction = (ITransaction*)pIUnknown;
	else if(riid == IID_IConnectionPointContainer) 
		m_pIConnectionPointContainer = (IConnectionPointContainer*)pIUnknown;
	else if(riid == IID_ISupportErrorInfo) 
		m_pISupportErrorInfo = (ISupportErrorInfo*)pIUnknown;
	else
		hr = E_NOINTERFACE;

	return hr;
}


/////////////////////////////////////////////////////////////////
// BOOL CTransaction::ReleaseObject
//
/////////////////////////////////////////////////////////////////
HRESULT CTransaction::ReleaseObject()
{
	//DataSource interfaces
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	pCListBox->OutputRelease((IUnknown**)&m_pITransaction,				"ITransaction");
	pCListBox->OutputRelease((IUnknown**)&m_pIConnectionPointContainer, "IConnectionPointContainer");
	pCListBox->OutputRelease((IUnknown**)&m_pISupportErrorInfo,			"ISupportErrorInfo");

	//Extra interfaces
	pCListBox->OutputRelease((IUnknown**)&m_pITransactionOptions,		"ITransactionOptions");

	//Base
	ReleaseBase("Transaction");
	return S_OK;
}


/////////////////////////////////////////////////////////////////
// HRESULT CTransaction::CreateObject
//
/////////////////////////////////////////////////////////////////
HRESULT CTransaction::CreateObject(IUnknown* pIUnknown)
{
	HRESULT hr = E_FAIL;
	HRESULT hrOptional = E_FAIL;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	COptionsDlg* pCOptionsDlg = GetOptionsObj();

	//Some don't return a rowset
	if(pIUnknown==NULL)
		goto CLEANUP;

	//Release any previous object
	ReleaseObject();

	//Now use the new object
	//[MANDATORY]
	XTESTC(hWnd, hr = pCListBox->OutputQI(pIUnknown, IID_IUnknown,		(IUnknown**)&m_pIUnknown));
	XTESTC(hWnd, hr = pCListBox->OutputQI(pIUnknown, IID_ITransaction,	(IUnknown**)&m_pITransaction));
	hrOptional = pCListBox->OutputQI(pIUnknown, IID_IConnectionPointContainer,	(IUnknown**)&m_pIConnectionPointContainer);
	
	//Auto QI
	if(pCOptionsDlg->m_dwRowsetOpts & ROWSET_AUTOQI)
	{
		//Obtain [optional] interfaces
		hrOptional = pCListBox->OutputQI(pIUnknown, IID_ISupportErrorInfo,	(IUnknown**)&m_pISupportErrorInfo);
	}

CLEANUP:
	return hr;
}



	
/////////////////////////////////////////////////////////////////
// CError::CError
//
/////////////////////////////////////////////////////////////////
CError::CError(CMDIChild* pCMDIChild) : CBase(pCMDIChild)
{
	//OLEDB Interfaces
	m_pIErrorInfo				= NULL;		//Error interface
	m_pIErrorRecords			= NULL;		//Error interface
	m_pISQLErrorInfo			= NULL;		//Error interface
}

/////////////////////////////////////////////////////////////////
// CError::~CError
//
/////////////////////////////////////////////////////////////////
CError::~CError()
{
	ReleaseError();
}


/////////////////////////////////////////////////////////////////
// HRESULT CError::SetInterface
//
/////////////////////////////////////////////////////////////////
HRESULT CError::SetInterface(REFIID riid, IUnknown* pIUnknown)
{
	HRESULT hr = S_OK;

	//no-op
	if(pIUnknown == NULL)
		return E_INVALIDARG;

	//If the requested interface is a Command interface, place it in the
	//Appropiate member variable, otherwise we have to release it, since we
	//have no place to store it...
	if(riid == IID_IUnknown) 
		m_pIUnknown = (IUnknown*)pIUnknown;
	else if(riid == IID_IErrorInfo) 
		m_pIErrorInfo = (IErrorInfo*)pIUnknown;
	else if(riid == IID_IErrorRecords) 
		m_pIErrorRecords = (IErrorRecords*)pIUnknown;
	else if(riid == IID_ISQLErrorInfo) 
		m_pISQLErrorInfo = (ISQLErrorInfo*)pIUnknown;
	else
		hr = E_NOINTERFACE;

	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CError::ReleaseError
//
/////////////////////////////////////////////////////////////////
HRESULT CError::ReleaseError()
{
	//Error
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	pCListBox->OutputRelease((IUnknown**)&m_pIErrorInfo,		"IErrorInfo");
	pCListBox->OutputRelease((IUnknown**)&m_pIErrorRecords,		"IErrorRecords");
	pCListBox->OutputRelease((IUnknown**)&m_pISQLErrorInfo,		"ISQLErrorInfo");
	ReleaseBase("Error");
	return S_OK;
}


/////////////////////////////////////////////////////////////////
// HRESULT CError::GetErrorInfo
//
/////////////////////////////////////////////////////////////////
HRESULT CError::GetErrorInfo()
{
	HRESULT				hr = S_OK;
	HRESULT				hrOptional = S_OK;
	
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	HWND hWnd = pCListBox->m_hWnd;

	//Release all current interfaces
	ReleaseError();

	//GetErrorInfo
	TESTC(pCListBox->OutputPreMethod("GetErrorInfo(0, &0x%08x)", m_pIUnknown));
	XTEST(hWnd, hr = ::GetErrorInfo(0, (IErrorInfo**)&m_pIUnknown));
	TESTC(pCListBox->OutputPostMethod(hr, "GetErrorInfo(0, &0x%08x)", m_pIUnknown));

	//AutoQI
	if(m_pIUnknown && GetOptionsObj()->m_dwRowsetOpts & ROWSET_AUTOQI)
	{
		//[MANDATORY]
		XTESTC(hWnd, hr = pCListBox->OutputQI(m_pIUnknown, IID_IErrorInfo,			(IUnknown**)&m_pIErrorInfo));
		XTESTC(hWnd, hr = pCListBox->OutputQI(m_pIUnknown, IID_IErrorRecords,		(IUnknown**)&m_pIErrorRecords));

		//[OPTIONAL]
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ISQLErrorInfo,			(IUnknown**)&m_pISQLErrorInfo);
	}

CLEANUP:
	return hr;
}



/////////////////////////////////////////////////////////////////
// HRESULT CError::SetErrorInfo
//
/////////////////////////////////////////////////////////////////
HRESULT CError::SetErrorInfo()
{
	HRESULT				hr = S_OK;
	HRESULT				hrOptional = S_OK;
	
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	HWND hWnd = pCListBox->m_hWnd;

	//SetErrorInfo
	TESTC(pCListBox->OutputPreMethod("SetErrorInfo(0, NULL)"));
	XTEST(hWnd, hr = ::SetErrorInfo(0, NULL));
	TESTC(pCListBox->OutputPostMethod(hr, "SetErrorInfo(0, NULL)"));

CLEANUP:
	return hr;
}



// {CB21F4D6-878D-11d1-9528-00C04FB66A50}
static const GUID IID_IAggregate = 
{ 0xcb21f4d6, 0x878d, 0x11d1, { 0x95, 0x28, 0x0, 0xc0, 0x4f, 0xb6, 0x6a, 0x50 } };

///////////////////////////////////////////////////////////////////////////////
// class CAggregate
// 
///////////////////////////////////////////////////////////////////////////////
CAggregate::CAggregate()
{
	m_cRef = 1;
	m_pIUnkInner = NULL;
}

CAggregate::~CAggregate() 
{
	//COM Aggregation rule #6
	//To free an inner pointer (other than IUnknown), the outer object calls its 
	//own outer unknown's AddRef followed by Release on the inner object's pointer
	//Currently we don't have this case, since we only have IUnknown
	//AddRef()
	//SAFE_RELEASE(m_pNonIUnknownInner);

	//Inner object free
	SAFE_RELEASE(m_pIUnkInner);
}

ULONG CAggregate::GetRefCount() 
{
	return m_cRef;
}

ULONG	CAggregate::AddRef(void)
{
	return ++m_cRef;
}

ULONG	CAggregate::Release(void)
{
	ASSERT(m_cRef);
	if(--m_cRef)
		return m_cRef;

	//COM Aggregation rule #5
	//The outer object must protect its implementation of Release from 
	//reentrantcy with an artifical reference count arround its destruction code
	m_cRef++;

	//Delete this object	
	delete this;
	return 0;
}

HRESULT CAggregate::QueryInterface(REFIID riid, LPVOID *ppv)
{
	HRESULT hr = S_OK;
	
	//TEST_ NULL
	if(ppv == NULL)
		return E_INVALIDARG;

	//Null output params
	*ppv = NULL;

	//Support IID_IUnknown
	if(riid == IID_IUnknown)
	{
		*ppv = (IUnknown*)this;
		SAFE_ADDREF((IUnknown*)*ppv);
	}
	else if(riid == IID_IAggregate)
	{
		*ppv = (IUnknown*)this;
		SAFE_ADDREF((IUnknown*)*ppv);
	}
	else if(m_pIUnkInner)
	{
		//Delegate the the Inner Object
		//This is not "circular" since this interface is the IID_IUnknown
		//interface only, which has its own non-delegating QI...
		hr = m_pIUnkInner->QueryInterface(riid, ppv);
	}
	else
	{
		return E_NOINTERFACE;
	}

	return hr;
}
